using UnityEngine;
using UnityEngine.UI;

public class TargetZone : MonoBehaviour
{
    private AnswerWithZone answer;
    private TargetZoneData zoneData;
    // Référence de résolution pour la conversion JSON -> écran (ex: 1920x1080)
    private static int referenceWidth = 1920;
    private static int referenceHeight = 1080;

    public static void SetReferenceResolution(int width, int height)
    {
        if (width > 0) referenceWidth = width;
        if (height > 0) referenceHeight = height;
    }

    public static void GetReferenceResolution(out int width, out int height)
    {
        width = referenceWidth;
        height = referenceHeight;
    }
    
    public void Initialize(AnswerWithZone answerData)
    {
        answer = answerData;
        zoneData = new TargetZoneData
        {
            id = answerData.zoneId,
            x = answerData.zone.x,
            y = answerData.zone.y,
            width = answerData.zone.width,
            height = answerData.zone.height
        };
        
        Debug.Log($"TargetZone initialisée: {answer.text} - Zone: ({zoneData.x},{zoneData.y},{zoneData.width},{zoneData.height})");
    }
    
    // Méthode principale pour vérifier si un point écran est dans cette zone
    public bool IsPointInScreenZone(Vector2 screenPoint)
    {
        if (zoneData == null) return false;
        
        // Vérifier si on est dans le nouveau layout de zone de jeu
        ShootingGameLayout layout = ShootingGameLayout.Instance;
        if (layout == null)
        {
            layout = FindFirstObjectByType<ShootingGameLayout>();
        }
        
        if (layout != null)
        {
            // Obtenir les limites de la zone de jeu en coordonnées écran
            Rect gameAreaScreen = layout.GetGameAreaScreenBounds();
            
            // Vérifier si le point est dans la zone de jeu
            if (screenPoint.x < gameAreaScreen.x || screenPoint.x > gameAreaScreen.x + gameAreaScreen.width ||
                screenPoint.y < gameAreaScreen.y || screenPoint.y > gameAreaScreen.y + gameAreaScreen.height)
            {
                return false; // Point hors de la zone de jeu
            }
            
            // Convertir les coordonnées écran vers les coordonnées relatives à la zone de jeu
            float relativeX = screenPoint.x - gameAreaScreen.x;
            float relativeY = screenPoint.y - gameAreaScreen.y;
            
            // Normaliser (0-1) puis convertir vers l'espace JSON (0-1920, 0-1080)
            float normalizedX = relativeX / gameAreaScreen.width;
            float normalizedY = relativeY / gameAreaScreen.height;
            
            // JSON a l'origine en haut-gauche, écran en bas-gauche
            float jsonX = normalizedX * referenceWidth;  // 0-1920
            float jsonY = (1f - normalizedY) * referenceHeight; // Inverser Y, 0-1080
            
            // Vérifier si le point est dans la zone
            bool inZone = jsonX >= zoneData.x && 
                         jsonX <= (zoneData.x + zoneData.width) &&
                         jsonY >= zoneData.y && 
                         jsonY <= (zoneData.y + zoneData.height);
            
            return inZone;
        }
        
        // FALLBACK : Ancien système (plein écran) si pas de layout
        float currentAspect = (float)Screen.width / Screen.height;
        float referenceAspect = (float)referenceWidth / referenceHeight;
        
        float fallbackScaleX, fallbackScaleY;
        float offsetX = 0, offsetY = 0;
        
        if (currentAspect > referenceAspect)
        {
            fallbackScaleY = (float)Screen.height / referenceHeight;
            fallbackScaleX = fallbackScaleY;
            float scaledWidth = referenceWidth * fallbackScaleX;
            offsetX = (Screen.width - scaledWidth) / 2f;
        }
        else if (currentAspect < referenceAspect)
        {
            fallbackScaleX = (float)Screen.width / referenceWidth;
            fallbackScaleY = fallbackScaleX;
            float scaledHeight = referenceHeight * fallbackScaleY;
            offsetY = (Screen.height - scaledHeight) / 2f;
        }
        else
        {
            fallbackScaleX = (float)Screen.width / referenceWidth;
            fallbackScaleY = (float)Screen.height / referenceHeight;
        }
        
        float fallbackJsonX = (screenPoint.x - offsetX) / fallbackScaleX;
        float fallbackJsonY = ((Screen.height - screenPoint.y) - offsetY) / fallbackScaleY;

        bool fallbackInZone = fallbackJsonX >= zoneData.x && 
                     fallbackJsonX <= (zoneData.x + zoneData.width) &&
                     fallbackJsonY >= zoneData.y && 
                     fallbackJsonY <= (zoneData.y + zoneData.height);
        
        return fallbackInZone;
    }
    
    public string GetAnswerText()
    {
        return answer?.text ?? "Réponse inconnue";
    }
    
    public bool IsCorrectAnswer()
    {
        return answer?.isCorrect ?? false;
    }
    
    public AnswerWithZone GetAnswer()
    {
        return answer;
    }
    
    // Méthode pour débugger les limites de la zone
    public void DebugZoneLimits()
    {
        if (answer != null && zoneData != null)
        {
            Debug.Log($"Zone '{answer.text}' limites JSON: X[{zoneData.x} à {zoneData.x + zoneData.width}] Y[{zoneData.y} à {zoneData.y + zoneData.height}]");
            Debug.Log($"Zone '{answer.text}' centre: ({zoneData.x + zoneData.width/2f}, {zoneData.y + zoneData.height/2f})");
        }
        else
        {
            Debug.LogWarning("DebugZoneLimits: answer ou zoneData est null");
        }
    }
}