using System.Collections;
using UnityEngine;
using UnityEngine.UI;
using TMPro;

/// <summary>
/// Gère la mise en page du jeu Trous (questions à trous)
/// Structure identique à ShootingGameLayout/CalculatorGameLayout mais SANS header
/// </summary>
public class TrousGameLayout : MonoBehaviour
{
    // Dimensions de référence (Canvas 1920x1080)
    [SerializeField] private float screenWidth = 1920f;
    [SerializeField] private float screenHeight = 1080f;
    [SerializeField] private float contentWidth = 1480f;    // Largeur zone de jeu
    [SerializeField] private float gameAreaHeight = 960f;   // Hauteur zone de jeu (sans header)
    [SerializeField] private float topMargin = 60f;         // Marge du haut
    [SerializeField] private float bottomMargin = 60f;      // Marge du bas
    [SerializeField] private float gameAreaCornerRadius = 40f; // Coins arrondis de la zone de jeu
    
    // Couleurs
    [SerializeField] private Color gameAreaBackgroundColor = new Color(0.96f, 0.93f, 0.90f, 1f); // #f5ece5
    
    // Références aux éléments UI
    private Canvas mainCanvas;
    private CanvasScaler canvasScaler;
    private RawImage backgroundImage; // Backdrop (plein écran)
    private RawImage gameAreaBackgroundImage; // Background DANS la zone de jeu
    private GameObject gameAreaPanel;
    private Texture2D loadedBackgroundTexture; // Texture du backdrop
    private Texture2D loadedGameAreaBackgroundTexture; // Texture du background de la zone de jeu
    private bool backdropSetByApi = false;
    private bool backdropLoadingComplete = false; // Flag pour savoir si le backdrop est chargé
    private bool isInitialized = false;
    private string backdropUrl = "";
    private string gameAreaBackgroundUrl = "";
    
    // Propriétés publiques
    public GameObject GameAreaPanel => gameAreaPanel;
    public RectTransform GameAreaRect => gameAreaPanel?.GetComponent<RectTransform>();
    public bool IsInitialized => isInitialized;
    public bool IsFullyLoaded => isInitialized && backdropLoadingComplete;
    
    // Singleton
    public static TrousGameLayout Instance { get; private set; }
    
    void Awake()
    {
        if (Instance == null)
        {
            Instance = this;
        }
        else
        {
            Destroy(gameObject);
            return;
        }
        
        SetupCanvas();
        CreateLayoutImmediate();
    }
    
    void Start()
    {
        // Charger l'image du backdrop si pas déjà défini par l'API
        if (!backdropSetByApi && !string.IsNullOrEmpty(backdropUrl))
        {
            StartCoroutine(LoadBackdropImageAsync());
        }
        else if (string.IsNullOrEmpty(backdropUrl))
        {
            Debug.Log("[TrousGameLayout] ℹ️ Pas de backdrop URL définie, utilisation de la couleur par défaut");
            // Pas de backdrop à charger, donc on est déjà "chargé"
            backdropLoadingComplete = true;
        }
        
        // Charger la config pour le corner radius si disponible
        StartCoroutine(TryLoadConfigDelayed());
    }
    
    IEnumerator TryLoadConfigDelayed()
    {
        // Attendre que GeneralConfigManager soit disponible et chargé
        float timeout = 3f;
        float elapsed = 0f;
        
        while ((GeneralConfigManager.Instance == null || !GeneralConfigManager.Instance.IsConfigLoaded()) && elapsed < timeout)
        {
            yield return new WaitForSeconds(0.1f);
            elapsed += 0.1f;
        }
        
        if (GeneralConfigManager.Instance != null && GeneralConfigManager.Instance.IsConfigLoaded())
        {
            var defaultUIConfig = GeneralConfigManager.Instance.GetDefaultUIConfig();
            if (defaultUIConfig?.bands != null)
            {
                // Mettre à jour le corner radius de la zone de jeu si nécessaire
                if (defaultUIConfig.bands.gameAreaCornerRadius > 0 && defaultUIConfig.bands.gameAreaCornerRadius != gameAreaCornerRadius)
                {
                    gameAreaCornerRadius = defaultUIConfig.bands.gameAreaCornerRadius;
                    Debug.Log($"[TrousGameLayout] ✅ Corner radius zone de jeu mis à jour: {gameAreaCornerRadius}px");
                    
                    // Mettre à jour les coins arrondis du GameAreaContainer
                    UpdateGameAreaCornerRadius();
                }
            }
        }
    }
    
    void UpdateGameAreaCornerRadius()
    {
        if (gameAreaPanel != null)
        {
            var roundedCorners = gameAreaPanel.GetComponent<RoundedCornersImage>();
            if (roundedCorners != null)
            {
                roundedCorners.cornerRadius = gameAreaCornerRadius;
                Debug.Log($"[TrousGameLayout] ✅ GameAreaPanel coins arrondis mis à jour: {gameAreaCornerRadius}px");
            }
            else
            {
                // Ajouter le composant si absent
                roundedCorners = gameAreaPanel.AddComponent<RoundedCornersImage>();
                roundedCorners.cornerRadius = gameAreaCornerRadius;
                Debug.Log($"[TrousGameLayout] ✅ RoundedCornersImage ajouté à GameAreaPanel: {gameAreaCornerRadius}px");
            }
        }
    }
    
    void SetupCanvas()
    {
        // Chercher un canvas existant
        mainCanvas = GetComponent<Canvas>();
        if (mainCanvas == null)
        {
            mainCanvas = GetComponentInParent<Canvas>();
        }
        if (mainCanvas == null)
        {
            mainCanvas = FindFirstObjectByType<Canvas>();
        }
        
        // Si pas de canvas trouvé, en créer un sur ce GameObject
        if (mainCanvas == null)
        {
            Debug.Log("[TrousGameLayout] Création d'un nouveau Canvas");
            mainCanvas = gameObject.AddComponent<Canvas>();
            mainCanvas.renderMode = RenderMode.ScreenSpaceOverlay;
            gameObject.AddComponent<GraphicRaycaster>();
        }
        
        // Configurer le CanvasScaler
        canvasScaler = mainCanvas.GetComponent<CanvasScaler>();
        if (canvasScaler == null)
        {
            canvasScaler = mainCanvas.gameObject.AddComponent<CanvasScaler>();
        }
        
        canvasScaler.uiScaleMode = CanvasScaler.ScaleMode.ScaleWithScreenSize;
        canvasScaler.referenceResolution = new Vector2(screenWidth, screenHeight);
        canvasScaler.screenMatchMode = CanvasScaler.ScreenMatchMode.MatchWidthOrHeight;
        canvasScaler.matchWidthOrHeight = 0.5f;
        
        Debug.Log($"[TrousGameLayout] Canvas configuré: {screenWidth}x{screenHeight}");
    }
    
    void CreateLayoutImmediate()
    {
        Debug.Log("[TrousGameLayout] ========== CRÉATION DU LAYOUT ==========");
        Debug.Log($"[TrousGameLayout] Canvas: {(mainCanvas != null ? mainCanvas.name : "NULL")}");
        Debug.Log($"[TrousGameLayout] Canvas RenderMode: {(mainCanvas != null ? mainCanvas.renderMode.ToString() : "N/A")}");
        
        // 1. Créer l'image de fond (backdrop - couvre tout l'écran)
        CreateBackgroundImage();
        
        // 2. Créer la zone de jeu centrale avec coins arrondis
        CreateGameAreaPanel();
        
        isInitialized = true;
        Debug.Log("[TrousGameLayout] ========== LAYOUT PRÊT ==========");
    }
    
    /// <summary>
    /// Crée l'image de fond plein écran (backdrop/habillage)
    /// </summary>
    void CreateBackgroundImage()
    {
        GameObject bgObj = new GameObject("MainBackgroundImage");
        bgObj.transform.SetParent(mainCanvas.transform, false);
        
        // FORCER en position 0 (tout premier)
        bgObj.transform.SetSiblingIndex(0);
        
        Debug.Log($"[TrousGameLayout] Backdrop - Sibling Index: {bgObj.transform.GetSiblingIndex()}, Total enfants Canvas: {mainCanvas.transform.childCount}");
        
        RectTransform bgRect = bgObj.AddComponent<RectTransform>();
        bgRect.anchorMin = Vector2.zero;
        bgRect.anchorMax = Vector2.one;
        bgRect.offsetMin = Vector2.zero;
        bgRect.offsetMax = Vector2.zero;
        
        backgroundImage = bgObj.AddComponent<RawImage>();
        backgroundImage.raycastTarget = false;
        // Couleur temporaire très visible pour débogage - sera remplacée par l'image
        backgroundImage.color = new Color(1f, 0.5f, 0f, 1f); // Orange vif pour test
        
        // PAS de Canvas enfant - juste SetAsFirstSibling() suffit pour mettre en arrière
        // Dans un Canvas ScreenSpaceOverlay, l'ordre dans la hiérarchie détermine l'ordre de rendu
        
        Debug.Log($"[TrousGameLayout] ✅ Backdrop créé: parent={mainCanvas.name}, pas de sortingOrder (premier dans hiérarchie), color={backgroundImage.color}");
        Debug.Log($"[TrousGameLayout] Backdrop GameObject actif: {bgObj.activeSelf}, position: {bgRect.anchoredPosition}");
    }
    
    /// <summary>
    /// Crée la zone de jeu centrale avec coins arrondis
    /// </summary>
    void CreateGameAreaPanel()
    {
        // === CONTENEUR PRINCIPAL AVEC COINS ARRONDIS (zone de jeu) ===
        gameAreaPanel = new GameObject("GameAreaContainer");
        gameAreaPanel.transform.SetParent(mainCanvas.transform, false);
        
        RectTransform containerRect = gameAreaPanel.AddComponent<RectTransform>();
        // Centrer verticalement et horizontalement
        containerRect.anchorMin = new Vector2(0.5f, 0.5f);
        containerRect.anchorMax = new Vector2(0.5f, 0.5f);
        containerRect.pivot = new Vector2(0.5f, 0.5f);
        containerRect.sizeDelta = new Vector2(contentWidth, gameAreaHeight);
        containerRect.anchoredPosition = Vector2.zero;
        
        // PAS de Canvas enfant non plus - l'ordre dans la hiérarchie suffit
        // Le gameAreaPanel sera après le backdrop dans la hiérarchie, donc devant
        gameAreaPanel.AddComponent<GraphicRaycaster>();
        
        // === IMAGE DE FOND DE LA ZONE DE JEU (optionnelle, chargée depuis l'API) ===
        GameObject bgImageObj = new GameObject("GameAreaBackground");
        bgImageObj.transform.SetParent(gameAreaPanel.transform, false);
        RectTransform bgImageRect = bgImageObj.AddComponent<RectTransform>();
        bgImageRect.anchorMin = Vector2.zero;
        bgImageRect.anchorMax = Vector2.one;
        bgImageRect.sizeDelta = Vector2.zero; // Remplit tout le parent
        bgImageRect.anchoredPosition = Vector2.zero;
        
        gameAreaBackgroundImage = bgImageObj.AddComponent<RawImage>();
        gameAreaBackgroundImage.raycastTarget = false;
        gameAreaBackgroundImage.enabled = false; // Désactivé par défaut (sera activé quand l'image sera chargée)
        bgImageObj.transform.SetAsFirstSibling(); // Au fond du GameAreaPanel
        
        Debug.Log("[TrousGameLayout] ✅ GameAreaBackground (RawImage) créé, désactivé par défaut");
        
        // Image de fond du conteneur (fallback si pas de background depuis l'API)
        Image containerBg = gameAreaPanel.AddComponent<Image>();
        containerBg.color = gameAreaBackgroundColor; // Beige clair
        containerBg.raycastTarget = true;
        
        // IMPORTANT: Ajouter les coins arrondis
        AddRoundedCorners(gameAreaPanel, gameAreaCornerRadius);
        
        // PAS DE MASK pour les jeux trous (pas besoin, pas de vidéo à découper)
        // Le Mask peut causer des problèmes de rendu avec les autres éléments
        
        Debug.Log($"[TrousGameLayout] ✅ GameArea créé:");
        Debug.Log($"  - Dimensions: {contentWidth}x{gameAreaHeight}");
        Debug.Log($"  - Parent: {mainCanvas.name}");
        Debug.Log($"  - Position: centré (anchoredPosition=0,0)");
        Debug.Log($"  - Couleur fond: {containerBg.color}, enabled: {containerBg.enabled}");
        Debug.Log($"  - Image GameObject actif: {gameAreaPanel.activeSelf}");
        Debug.Log($"  - Sibling index: {gameAreaPanel.transform.GetSiblingIndex()}");
        Debug.Log($"  - Coins arrondis: {gameAreaCornerRadius}px");
        Debug.Log($"[TrousGameLayout] 📐 Écran: {Screen.width}x{Screen.height}");
        Debug.Log($"[TrousGameLayout] 📐 Canvas réf: {screenWidth}x{screenHeight}");
        Debug.Log($"[TrousGameLayout] 📐 Marges visibles: {(screenWidth - contentWidth) / 2f}px (gauche/droite), {(screenHeight - gameAreaHeight) / 2f}px (haut/bas)");
    }
    
    void AddRoundedCorners(GameObject target, float radius)
    {
        RoundedCornersImage roundedCorners = target.GetComponent<RoundedCornersImage>();
        if (roundedCorners == null)
        {
            roundedCorners = target.AddComponent<RoundedCornersImage>();
        }
        roundedCorners.cornerRadius = radius;
    }
    
    /// <summary>
    /// Définit l'URL du backdrop (appelé depuis l'API ou la config)
    /// </summary>
    public void SetBackdropUrl(string url, bool fromApi = false)
    {
        if (backdropSetByApi && !fromApi)
        {
            Debug.Log($"[TrousGameLayout] ⏭️ Backdrop ignoré (déjà défini par l'API): {url}");
            return;
        }
        
        if (fromApi)
        {
            backdropSetByApi = true;
            Debug.Log($"[TrousGameLayout] 🎯 Backdrop défini par l'API: {url}");
        }
        
        backdropUrl = url;
        
        // Reset backdropLoadingComplete si on change d'URL
        if (!string.IsNullOrEmpty(url))
        {
            backdropLoadingComplete = false;
        }
        
        if (gameObject.activeInHierarchy && !string.IsNullOrEmpty(url))
        {
            StartCoroutine(LoadBackdropImageAsync());
        }
        else if (string.IsNullOrEmpty(url))
        {
            // Pas de backdrop, donc on est "chargé"
            backdropLoadingComplete = true;
        }
    }
    
    /// <summary>
    /// Charge l'image de backdrop de manière asynchrone
    /// </summary>
    IEnumerator LoadBackdropImageAsync()
    {
        if (string.IsNullOrEmpty(backdropUrl) || backgroundImage == null)
        {
            backdropLoadingComplete = true;
            yield break;
        }
        
        Debug.Log($"[TrousGameLayout] 🖼️ Chargement du backdrop (habillage): {backdropUrl}");
        
        // 🚀 NOUVEAU : Vérifier d'abord le cache de GameDataManager
        if (GameDataManager.Instance != null)
        {
            Texture2D cachedTexture = GameDataManager.Instance.GetCachedTexture("backdrop");
            if (cachedTexture != null)
            {
                Debug.Log($"[TrousGameLayout] ⚡ Backdrop chargé instantanément depuis le cache !");
                loadedBackgroundTexture = cachedTexture;
                backgroundImage.texture = cachedTexture;
                backdropLoadingComplete = true;
                yield break;
            }
        }
        
        // Résoudre l'URL si nécessaire
        string resolvedUrl = backdropUrl;
        if (GeneralConfigManager.Instance != null)
        {
            resolvedUrl = GeneralConfigManager.Instance.GetBackgroundImageUrl(backdropUrl);
        }
        
        using (UnityEngine.Networking.UnityWebRequest www = UnityEngine.Networking.UnityWebRequestTexture.GetTexture(resolvedUrl))
        {
            yield return www.SendWebRequest();
            
            if (www.result == UnityEngine.Networking.UnityWebRequest.Result.Success)
            {
                loadedBackgroundTexture = UnityEngine.Networking.DownloadHandlerTexture.GetContent(www);
                
                if (backgroundImage != null && loadedBackgroundTexture != null)
                {
                    backgroundImage.texture = loadedBackgroundTexture;
                    backgroundImage.color = Color.white; // Blanc pour afficher la texture normalement
                    
                    Debug.Log($"[TrousGameLayout] ✅ Backdrop chargé et appliqué: {loadedBackgroundTexture.width}x{loadedBackgroundTexture.height}");
                    Debug.Log($"[TrousGameLayout] RawImage texture: {backgroundImage.texture != null}, color: {backgroundImage.color}, enabled: {backgroundImage.enabled}");
                }
                else
                {
                    Debug.LogError($"[TrousGameLayout] ❌ backgroundImage ou texture null après chargement");
                }
            }
            else
            {
                Debug.LogError($"[TrousGameLayout] ❌ Erreur chargement backdrop: {www.error}");
            }
            
            // Toujours marquer comme chargé, même en cas d'erreur
            backdropLoadingComplete = true;
            Debug.Log($"[TrousGameLayout] ✅ Backdrop loading complete - IsFullyLoaded: {IsFullyLoaded}");
        }
    }
    
    /// <summary>
    /// Définit l'URL du background de la zone de jeu (appelé depuis l'API)
    /// </summary>
    public void SetGameAreaBackgroundUrl(string url)
    {
        gameAreaBackgroundUrl = url;
        
        if (gameObject.activeInHierarchy && !string.IsNullOrEmpty(url))
        {
            StartCoroutine(LoadGameAreaBackgroundImageAsync());
        }
    }
    
    /// <summary>
    /// Charge l'image de background de la zone de jeu de manière asynchrone
    /// </summary>
    IEnumerator LoadGameAreaBackgroundImageAsync()
    {
        if (string.IsNullOrEmpty(gameAreaBackgroundUrl) || gameAreaBackgroundImage == null) yield break;
        
        Debug.Log($"[TrousGameLayout] 🖼️ Chargement du background de la zone de jeu: {gameAreaBackgroundUrl}");
        
        // Résoudre l'URL si nécessaire
        string resolvedUrl = gameAreaBackgroundUrl;
        if (GeneralConfigManager.Instance != null)
        {
            resolvedUrl = GeneralConfigManager.Instance.GetBackgroundImageUrl(gameAreaBackgroundUrl);
        }
        
        using (UnityEngine.Networking.UnityWebRequest www = UnityEngine.Networking.UnityWebRequestTexture.GetTexture(resolvedUrl))
        {
            yield return www.SendWebRequest();
            
            if (www.result == UnityEngine.Networking.UnityWebRequest.Result.Success)
            {
                loadedGameAreaBackgroundTexture = UnityEngine.Networking.DownloadHandlerTexture.GetContent(www);
                
                if (gameAreaBackgroundImage != null && loadedGameAreaBackgroundTexture != null)
                {
                    gameAreaBackgroundImage.texture = loadedGameAreaBackgroundTexture;
                    gameAreaBackgroundImage.color = Color.white; // Blanc pour afficher la texture normalement
                    gameAreaBackgroundImage.enabled = true; // Activer l'image
                    
                    Debug.Log($"[TrousGameLayout] ✅ Background de la zone de jeu chargé et appliqué: {loadedGameAreaBackgroundTexture.width}x{loadedGameAreaBackgroundTexture.height}");
                }
                else
                {
                    Debug.LogError($"[TrousGameLayout] ❌ gameAreaBackgroundImage ou texture null après chargement");
                }
            }
            else
            {
                Debug.LogError($"[TrousGameLayout] ❌ Erreur chargement background zone de jeu: {www.error}");
            }
        }
    }
    
    /// <summary>
    /// Obtient les dimensions de la zone de jeu
    /// </summary>
    public Vector2 GetGameAreaSize()
    {
        return new Vector2(contentWidth, gameAreaHeight);
    }
    
    void OnDestroy()
    {
        if (loadedBackgroundTexture != null)
        {
            Destroy(loadedBackgroundTexture);
        }
        
        if (Instance == this)
        {
            Instance = null;
        }
    }
}
