using System.Collections;
using System.Collections.Generic;
using System.IO;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Networking;
using TMPro;
using UnityEngine.EventSystems;

/// <summary>
/// Menu principal - Mosaïque de jeux chargée depuis le CSV
/// Charge StreamingAssets/json/jeux.csv et affiche tous les jeux disponibles
/// </summary>
public class MainMenuManager : MonoBehaviour
{
    [Header("UI References")]
    public TextMeshProUGUI titleText;

    [Tooltip("Viewport du ScrollRect (enfant direct de l'objet Scroll View)")]
    public RectTransform viewport;

    [Tooltip("Transform/RectTransform qui contient les cartes (Content du ScrollRect)")]
    public RectTransform levelsContainer;

    [Tooltip("Scroll View (ScrollRect)")]
    public ScrollRect scrollRect;

    [Tooltip("Scrollbar verticale (optionnelle)")]
    public Scrollbar verticalScrollbar;

    [Header("Loading")]
    public GameObject loadingPanel;
    public TextMeshProUGUI loadingText;

    [Header("Layout (Grid)")]
    [SerializeField] private int columns = 6;
    [SerializeField] private Vector2 cellSize = new Vector2(290, 120);
    [SerializeField] private Vector2 spacing = new Vector2(10, 10);
    [SerializeField] private RectOffset padding;
    [SerializeField] private TextAnchor childAlignment = TextAnchor.UpperLeft;

    [Header("Card Style")]
    [SerializeField] private Color shootingColor = new Color(0.39f, 0.28f, 0.5f, 1f);
    [SerializeField] private Color textHoleColor = new Color(0.8f, 0.58f, 0.26f, 1f);
    [SerializeField] private Color calculatorColor = new Color(0.33f, 0.61f, 0.36f, 1f);
    [SerializeField] private Color defaultColor = new Color(0.5f, 0.5f, 0.5f, 1f);
    [SerializeField] private int titleFontSize = 18;
    [SerializeField] private int typeFontSize = 14;

    [Header("Debug")]
    public bool debugMode = false;

    // Données internes
    private readonly List<GameEntry> gameEntries = new List<GameEntry>();
    private readonly List<GameObject> gameCards = new List<GameObject>();
    private GridLayoutGroup grid;
    private string csvPath;

    // Gate "anti écran noir" : tant que la config générale n'est pas exploitable, on bloque les clics sur les jeux.
    private Button retryConfigButton;

    void Awake()
    {
        if (padding == null) padding = new RectOffset(20, 20, 20, 20);

        EnsureEventSystem();

        // Chemin du fichier CSV
        // IMPORTANT WebGL: Application.streamingAssetsPath est une URL HTTP. Sur Windows, Path.Combine injecte des '\'
        // qui cassent l'URL (=> UnityWebRequest échoue). On normalise donc toujours en '/'.
        csvPath = Path.Combine(Application.streamingAssetsPath, "json", "jeux.csv").Replace("\\", "/");

        if (levelsContainer == null)
        {
            Debug.LogError("[MainMenuManager] 'levelsContainer' n'est pas assigné.");
            return;
        }

        // Configurer le GridLayoutGroup sur le Content
        grid = levelsContainer.GetComponent<GridLayoutGroup>();
        if (grid == null) grid = levelsContainer.gameObject.AddComponent<GridLayoutGroup>();

        // Configure le Grid
        grid.cellSize = cellSize;
        grid.spacing = spacing;
        grid.padding = padding;
        grid.startCorner = GridLayoutGroup.Corner.UpperLeft;
        grid.startAxis = GridLayoutGroup.Axis.Horizontal;
        grid.childAlignment = childAlignment;
        grid.constraint = GridLayoutGroup.Constraint.FixedColumnCount;
        grid.constraintCount = Mathf.Max(1, columns);

        // Ancrage du Content pour le scroll vertical
        levelsContainer.anchorMin = new Vector2(0f, 1f);
        levelsContainer.anchorMax = new Vector2(1f, 1f);
        levelsContainer.pivot = new Vector2(0.5f, 1f);
        levelsContainer.anchoredPosition = Vector2.zero;
    }

    private void EnsureEventSystem()
    {
        // IMPORTANT: Sans EventSystem, aucun clic UI ne marche (ça ressemble à un "voile" qui bloque tout).
        if (EventSystem.current != null) return;

        GameObject eventSystemObj = new GameObject("EventSystem");
        eventSystemObj.AddComponent<EventSystem>();
#if ENABLE_INPUT_SYSTEM
        eventSystemObj.AddComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
#else
        eventSystemObj.AddComponent<StandaloneInputModule>();
#endif
        DontDestroyOnLoad(eventSystemObj);
    }

    void Start()
    {
        if (loadingPanel != null)
        {
            loadingPanel.SetActive(true);
            if (loadingText != null) loadingText.text = "Chargement des jeux...";
        }

        SetupScrollRect();
        StartCoroutine(InitializeMenu());
    }

    /// <summary>
    /// Initialise le menu : charge la config, vérifie la connexion, crée le header et charge les jeux
    /// </summary>
    private IEnumerator InitializeMenu()
    {
        // Attendre que GeneralConfigManager soit prêt
        if (GeneralConfigManager.Instance != null)
        {
            yield return GeneralConfigManager.Instance.WaitForConfigLoaded();
        }
        else
        {
            Debug.LogError("[MainMenuManager] GeneralConfigManager.Instance est null!");
            yield break;
        }

        // IMPORTANT: IsConfigLoaded peut être true même si on est tombé sur une config vide (fallback).
        // Dans ce cas, on bloque le menu et on propose un "Réessayer" au lieu de laisser partir vers un écran noir.
        UpdateConfigGateUI();

        // Vérifier le statut de connexion
        yield return StartCoroutine(CheckLoginStatusAndShowPopup());

        // Créer le header
        yield return StartCoroutine(CreateHeader());

        // Charger et afficher les jeux depuis le CSV
        yield return StartCoroutine(LoadGamesFromCSV());

        // Masquer le panneau de chargement seulement si la config est OK
        if (loadingPanel != null && IsGeneralConfigUsable())
        {
            loadingPanel.SetActive(false);
        }
    }

    /// <summary>
    /// Charge les jeux depuis le fichier CSV
    /// </summary>
    private IEnumerator LoadGamesFromCSV()
    {
        Debug.Log($"[MainMenuManager] 📄 Chargement du CSV: {csvPath}");

        string csvContent = "";

        // Charger le fichier CSV
        if (csvPath.Contains("://") || csvPath.Contains(":///"))
        {
            // Sur Android/WebGL, utiliser UnityWebRequest
            using (UnityWebRequest www = UnityWebRequest.Get(csvPath))
            {
                yield return www.SendWebRequest();

                if (www.result != UnityWebRequest.Result.Success)
                {
                    Debug.LogError($"[MainMenuManager] ❌ Erreur chargement CSV: {www.error}");
                    yield break;
                }

                csvContent = www.downloadHandler.text;
            }
        }
        else
        {
            // Sur desktop, lecture directe
            if (!File.Exists(csvPath))
            {
                Debug.LogError($"[MainMenuManager] ❌ Fichier CSV introuvable: {csvPath}");
                yield break;
            }

            csvContent = File.ReadAllText(csvPath);
        }

        // Parser le CSV
        ParseCSV(csvContent);

        // Générer les cartes
        GenerateGameCards();

        // Gate: si config invalide, désactiver les clics et afficher un message
        ApplyCardsInteractivity(IsGeneralConfigUsable());
        UpdateConfigGateUI();

        // Mettre à jour le layout
        yield return StartCoroutine(UpdateLayoutWithDelay());
    }

    /// <summary>
    /// Parse le contenu CSV et remplit la liste des jeux
    /// Format: id,title,type
    /// </summary>
    private void ParseCSV(string csvContent)
    {
        gameEntries.Clear();

        // Supprimer le BOM UTF-8 si présent
        if (csvContent.Length > 0 && csvContent[0] == '\uFEFF')
        {
            csvContent = csvContent.Substring(1);
            Debug.Log("[MainMenuManager] 📄 BOM UTF-8 supprimé");
        }

        string[] lines = csvContent.Split(new[] { '\r', '\n' }, System.StringSplitOptions.RemoveEmptyEntries);

        Debug.Log($"[MainMenuManager] 📊 {lines.Length} lignes dans le CSV");
        
        // Afficher les premières lignes pour debug
        for (int dbg = 0; dbg < Mathf.Min(3, lines.Length); dbg++)
        {
            Debug.Log($"[MainMenuManager] 📄 Ligne {dbg}: '{lines[dbg]}'");
        }

        for (int i = 0; i < lines.Length; i++)
        {
            string line = lines[i].Trim();
            
            // Ignorer les lignes vides
            if (string.IsNullOrEmpty(line)) continue;

            // Ignorer la ligne d'en-tête
            if (i == 0 && (line.ToLower().Contains("\"id\"") || line.StartsWith("id,") || line.StartsWith("\"id\"")))
            {
                Debug.Log($"[MainMenuManager] ⏭️ Ligne d'en-tête ignorée: {line}");
                continue;
            }

            // Parser la ligne CSV (gestion des guillemets)
            List<string> fields = ParseCSVLine(line);

            if (fields.Count >= 3)
            {
                string idStr = fields[0].Trim().Trim('"');
                
                if (int.TryParse(idStr, out int id))
                {
                    GameEntry entry = new GameEntry
                    {
                        id = id,
                        title = fields[1].Trim('"'),
                        type = fields[2].Trim('"')
                    };

                    gameEntries.Add(entry);

                    // Toujours logger les 3 premiers pour debug
                    if (gameEntries.Count <= 3 || debugMode)
                        Debug.Log($"[MainMenuManager] ✅ Jeu {gameEntries.Count}: ID={entry.id}, '{entry.title}' ({entry.type})");
                }
                else
                {
                    Debug.LogWarning($"[MainMenuManager] ⚠️ ID invalide ligne {i + 1}: '{idStr}' (raw: '{fields[0]}')");
                }
            }
            else
            {
                Debug.LogWarning($"[MainMenuManager] ⚠️ Ligne {i + 1} n'a que {fields.Count} champs: '{line}'");
            }
        }

        Debug.Log($"[MainMenuManager] ✅ {gameEntries.Count} jeux chargés depuis le CSV");
    }

    /// <summary>
    /// Parse une ligne CSV en gérant les guillemets
    /// </summary>
    private List<string> ParseCSVLine(string line)
    {
        List<string> fields = new List<string>();
        bool inQuotes = false;
        string currentField = "";

        for (int i = 0; i < line.Length; i++)
        {
            char c = line[i];

            if (c == '"')
            {
                inQuotes = !inQuotes;
            }
            else if (c == ',' && !inQuotes)
            {
                fields.Add(currentField);
                currentField = "";
            }
            else
            {
                currentField += c;
            }
        }

        // Ajouter le dernier champ
        fields.Add(currentField);

        return fields;
    }

    /// <summary>
    /// Génère les cartes de jeu dans la mosaïque
    /// </summary>
    private void GenerateGameCards()
    {
        ClearGameCards();

        if (levelsContainer == null)
        {
            Debug.LogError("[MainMenuManager] levelsContainer est null !");
            return;
        }

        foreach (var entry in gameEntries)
        {
            if (entry == null) continue;
            
            GameObject cardObj = CreateGameCard(entry);
            if (cardObj != null)
            {
                gameCards.Add(cardObj);
            }
        }

        Debug.Log($"[MainMenuManager] ✅ {gameCards.Count} cartes générées sur {gameEntries.Count} jeux");
    }

    private bool IsGeneralConfigUsable()
    {
        return GeneralConfigManager.Instance != null &&
               GeneralConfigManager.Instance.IsConfigLoaded() &&
               GeneralConfigManager.Instance.HasValidApiUrls();
    }

    private void ApplyCardsInteractivity(bool enabled)
    {
        foreach (var cardObj in gameCards)
        {
            if (cardObj == null) continue;

            var btn = cardObj.GetComponent<Button>();
            if (btn != null) btn.interactable = enabled;

            var cg = cardObj.GetComponent<CanvasGroup>();
            if (cg != null)
            {
                cg.alpha = enabled ? 1f : 0.5f;
                cg.blocksRaycasts = enabled;
                cg.interactable = enabled;
            }
        }
    }

    private void UpdateConfigGateUI()
    {
        if (loadingPanel == null) return;

        bool ok = IsGeneralConfigUsable();
        if (ok)
        {
            if (retryConfigButton != null) retryConfigButton.gameObject.SetActive(false);
            return;
        }

        loadingPanel.SetActive(true);
        if (loadingText != null)
        {
            loadingText.text =
                "Configuration non chargée.\n\n" +
                "Ceci arrive généralement à cause du cache navigateur/serveur.\n" +
                "Clique sur « Réessayer » ou fais un rechargement forcé (Ctrl+F5).";
        }

        EnsureRetryButtonExists();
        if (retryConfigButton != null)
        {
            retryConfigButton.gameObject.SetActive(true);
            retryConfigButton.interactable = true;
        }

        ApplyCardsInteractivity(false);
    }

    private void EnsureRetryButtonExists()
    {
        if (retryConfigButton != null || loadingPanel == null) return;

        GameObject btnObj = new GameObject("RetryConfigButton");
        btnObj.transform.SetParent(loadingPanel.transform, false);

        var rt = btnObj.AddComponent<RectTransform>();
        rt.anchorMin = new Vector2(0.5f, 0.5f);
        rt.anchorMax = new Vector2(0.5f, 0.5f);
        rt.pivot = new Vector2(0.5f, 0.5f);
        rt.sizeDelta = new Vector2(260, 60);
        rt.anchoredPosition = new Vector2(0, -90);

        var img = btnObj.AddComponent<Image>();
        img.color = new Color(0.15f, 0.55f, 0.9f, 1f);

        retryConfigButton = btnObj.AddComponent<Button>();
        retryConfigButton.targetGraphic = img;
        retryConfigButton.transition = Selectable.Transition.ColorTint;

        // Texte
        GameObject labelObj = new GameObject("Label");
        labelObj.transform.SetParent(btnObj.transform, false);
        var labelRt = labelObj.AddComponent<RectTransform>();
        labelRt.anchorMin = Vector2.zero;
        labelRt.anchorMax = Vector2.one;
        labelRt.offsetMin = Vector2.zero;
        labelRt.offsetMax = Vector2.zero;

        var tmp = labelObj.AddComponent<TextMeshProUGUI>();
        tmp.text = "RÉESSAYER";
        tmp.alignment = TextAlignmentOptions.Center;
        tmp.fontSize = 22;
        tmp.color = Color.white;
        // Assurer une police visible même si la config générale n'est pas chargée
        if (tmp.font == null)
        {
            tmp.font = Resources.Load<TMP_FontAsset>("Fonts/Lato-Regular SDF") ?? TMP_Settings.defaultFontAsset;
        }

        retryConfigButton.onClick.RemoveAllListeners();
        retryConfigButton.onClick.AddListener(() =>
        {
            Debug.Log("[MainMenuManager] 🔄 Réessayer: ReloadConfig()");
            retryConfigButton.interactable = false;
            if (loadingText != null) loadingText.text = "Rechargement de la configuration…";

            if (GeneralConfigManager.Instance != null)
            {
                GeneralConfigManager.Instance.ReloadConfig();
                StartCoroutine(WaitForConfigThenEnableMenu());
            }
        });
    }

    private IEnumerator WaitForConfigThenEnableMenu()
    {
        if (GeneralConfigManager.Instance != null)
        {
            yield return GeneralConfigManager.Instance.WaitForConfigLoaded();
        }

        bool ok = IsGeneralConfigUsable();
        if (ok)
        {
            ApplyCardsInteractivity(true);
            if (loadingPanel != null) loadingPanel.SetActive(false);
        }
        else
        {
            UpdateConfigGateUI();
        }
    }

    /// <summary>
    /// Crée une carte de jeu programmatiquement
    /// </summary>
    private GameObject CreateGameCard(GameEntry entry)
    {
        if (entry == null)
        {
            Debug.LogError("[MainMenuManager] ❌ Entry est null dans CreateGameCard!");
            return null;
        }

        try
        {
            // Créer l'objet carte
            GameObject cardObj = new GameObject($"GameCard_{entry.id}");
            cardObj.transform.SetParent(levelsContainer, false);

            // RectTransform
            RectTransform rt = cardObj.AddComponent<RectTransform>();
            rt.sizeDelta = cellSize;
            rt.localScale = Vector3.one;

            // Image de fond (avant CanvasGroup et Button)
            Image bgImage = cardObj.AddComponent<Image>();
            bgImage.color = GetTypeColor(entry.type);
            bgImage.raycastTarget = true;

            // CanvasGroup pour les effets
            CanvasGroup canvasGroup = cardObj.AddComponent<CanvasGroup>();

            // Ajouter le bouton AVANT GameCard
            Button button = cardObj.AddComponent<Button>();
            button.targetGraphic = bgImage;
            
            // Configurer les transitions du bouton
            ColorBlock colors = button.colors;
            colors.normalColor = Color.white;
            colors.highlightedColor = new Color(1.1f, 1.1f, 1.1f, 1f);
            colors.pressedColor = new Color(0.9f, 0.9f, 0.9f, 1f);
            button.colors = colors;

            // Container pour le contenu
            GameObject contentContainer = new GameObject("Content");
            contentContainer.transform.SetParent(cardObj.transform, false);
            RectTransform contentRt = contentContainer.AddComponent<RectTransform>();
            contentRt.anchorMin = Vector2.zero;
            contentRt.anchorMax = Vector2.one;
            contentRt.offsetMin = new Vector2(10, 10);
            contentRt.offsetMax = new Vector2(-10, -10);

            // Titre du jeu
            GameObject titleObj = new GameObject("Title");
            titleObj.transform.SetParent(contentContainer.transform, false);
            RectTransform titleRt = titleObj.AddComponent<RectTransform>();
            titleRt.anchorMin = new Vector2(0, 0.3f);
            titleRt.anchorMax = new Vector2(1, 1);
            titleRt.offsetMin = Vector2.zero;
            titleRt.offsetMax = Vector2.zero;

            TextMeshProUGUI titleText = titleObj.AddComponent<TextMeshProUGUI>();
            titleText.text = entry.title;
            titleText.fontSize = titleFontSize;
            titleText.color = Color.white;
            titleText.alignment = TextAlignmentOptions.Center;
            titleText.textWrappingMode = TextWrappingModes.Normal;
            titleText.overflowMode = TextOverflowModes.Ellipsis;

            // Type du jeu
            GameObject typeObj = new GameObject("Type");
            typeObj.transform.SetParent(contentContainer.transform, false);
            RectTransform typeRt = typeObj.AddComponent<RectTransform>();
            typeRt.anchorMin = new Vector2(0, 0);
            typeRt.anchorMax = new Vector2(1, 0.3f);
            typeRt.offsetMin = Vector2.zero;
            typeRt.offsetMax = Vector2.zero;

            TextMeshProUGUI typeText = typeObj.AddComponent<TextMeshProUGUI>();
            typeText.text = entry.GetDisplayType();
            typeText.fontSize = typeFontSize;
            typeText.color = new Color(1f, 1f, 1f, 0.7f);
            typeText.alignment = TextAlignmentOptions.Center;

            // Ajouter le composant GameCard EN DERNIER
            GameCard gameCard = cardObj.AddComponent<GameCard>();
            gameCard.titleText = titleText;
            gameCard.typeText = typeText;
            gameCard.backgroundImage = bgImage;
            gameCard.cardButton = button;
            gameCard.shootingColor = shootingColor;
            gameCard.textHoleColor = textHoleColor;
            gameCard.calculatorColor = calculatorColor;
            gameCard.defaultColor = defaultColor;

            // Initialiser la carte avec les données
            gameCard.Initialize(entry);

            return cardObj;
        }
        catch (System.Exception e)
        {
            Debug.LogError($"[MainMenuManager] ❌ Erreur création carte pour {entry?.title}: {e.Message}");
            Debug.LogError($"[MainMenuManager] Stack: {e.StackTrace}");
            return null;
        }
    }

    /// <summary>
    /// Retourne la couleur associée au type de jeu
    /// </summary>
    private Color GetTypeColor(string type)
    {
        switch (type?.ToLower())
        {
            case "shooting": return shootingColor;
            case "text_hole": return textHoleColor;
            case "calculator": return calculatorColor;
            default: return defaultColor;
        }
    }

    /// <summary>
    /// Supprime toutes les cartes existantes
    /// </summary>
    private void ClearGameCards()
    {
        foreach (var card in gameCards)
        {
            if (card != null) Destroy(card);
        }
        gameCards.Clear();
    }

    // ========================================
    // SCROLL & LAYOUT
    // ========================================

    void SetupScrollRect()
    {
        if (scrollRect == null)
        {
            Debug.LogError("[MainMenuManager] ScrollRect non assigné.");
            return;
        }

        scrollRect.content = levelsContainer;
        if (viewport != null) scrollRect.viewport = viewport;

        if (viewport != null && viewport.GetComponent<RectMask2D>() == null)
            viewport.gameObject.AddComponent<RectMask2D>();

        scrollRect.horizontal = false;
        scrollRect.vertical = true;
        scrollRect.movementType = ScrollRect.MovementType.Clamped;
        scrollRect.inertia = true;
        scrollRect.scrollSensitivity = 25f;
        scrollRect.decelerationRate = 0.135f;

        if (verticalScrollbar != null)
        {
            scrollRect.verticalScrollbar = verticalScrollbar;
#if UNITY_2021_1_OR_NEWER
            scrollRect.verticalScrollbarVisibility = ScrollRect.ScrollbarVisibility.AutoHideAndExpandViewport;
#endif
        }
    }

    IEnumerator UpdateLayoutWithDelay()
    {
        yield return new WaitForEndOfFrame();
        yield return new WaitForEndOfFrame();

        UpdateContentHeight();
        ScrollToTop();

        LayoutRebuilder.ForceRebuildLayoutImmediate(levelsContainer);
        Canvas.ForceUpdateCanvases();
    }

    void UpdateContentHeight()
    {
        if (grid == null || levelsContainer == null) return;

        int items = gameCards.Count;
        if (items == 0)
        {
            levelsContainer.SetSizeWithCurrentAnchors(RectTransform.Axis.Vertical, 100f);
            return;
        }

        int cols = Mathf.Max(1, columns);
        int rows = Mathf.CeilToInt(items / (float)cols);

        float totalHeight = padding.top
                          + rows * cellSize.y
                          + Mathf.Max(0, rows - 1) * spacing.y
                          + padding.bottom;

        float viewportHeight = viewport ? viewport.rect.height : 600f;
        float minHeight = Mathf.Max(totalHeight, viewportHeight + 50f);

        levelsContainer.SetSizeWithCurrentAnchors(RectTransform.Axis.Vertical, minHeight);
        LayoutRebuilder.ForceRebuildLayoutImmediate(levelsContainer);

        if (scrollRect != null)
        {
            scrollRect.Rebuild(CanvasUpdate.PostLayout);
        }
    }

    void ScrollToTop()
    {
        if (scrollRect != null)
        {
            scrollRect.verticalNormalizedPosition = 1f;
            scrollRect.velocity = Vector2.zero;
        }
    }

    // ========================================
    // HEADER & LOGIN
    // ========================================

    private IEnumerator CreateHeader()
    {
        Canvas canvas = FindFirstObjectByType<Canvas>();
        if (canvas == null)
        {
            Debug.LogError("[MainMenuManager] Canvas introuvable pour créer le header");
            yield break;
        }

        DefaultHeaderConfig headerConfig = GeneralConfigManager.Instance.GetDefaultHeaderConfig();
        if (headerConfig == null || headerConfig.rightElement == null)
        {
            Debug.LogError("[MainMenuManager] Configuration header introuvable");
            yield break;
        }

        if (UnityEngine.EventSystems.EventSystem.current == null)
        {
            GameObject eventSystemObj = new GameObject("EventSystem");
            eventSystemObj.AddComponent<UnityEngine.EventSystems.EventSystem>();
#if ENABLE_INPUT_SYSTEM
            eventSystemObj.AddComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
#else
            eventSystemObj.AddComponent<UnityEngine.EventSystems.StandaloneInputModule>();
#endif
        }

        if (canvas.GetComponent<GraphicRaycaster>() == null)
        {
            canvas.gameObject.AddComponent<GraphicRaycaster>();
        }

        HeaderManager.Instance.SetHeaderElementClickCallback(OnHeaderElementClick);

        GameObject headerPanel = new GameObject("HeaderPanel");
        headerPanel.transform.SetParent(canvas.transform, false);

        RectTransform headerRect = headerPanel.AddComponent<RectTransform>();
        headerRect.anchorMin = new Vector2(0f, 1f);
        headerRect.anchorMax = new Vector2(1f, 1f);
        headerRect.pivot = new Vector2(0.5f, 1f);
        headerRect.sizeDelta = new Vector2(0f, headerConfig.height);
        headerRect.anchoredPosition = Vector2.zero;

        Image headerBg = headerPanel.AddComponent<Image>();
        if (ColorUtility.TryParseHtmlString(headerConfig.backgroundColor, out Color bgColor))
        {
            headerBg.color = bgColor;
        }
        headerBg.raycastTarget = false;

        DefaultHeaderElement rightElement = headerConfig.rightElement;
        DefaultHeaderElement modifiedRightElement = new DefaultHeaderElement
        {
            imageUrl = rightElement.imageUrl,
            position = rightElement.position,
            size = new Vector2(rightElement.size.x / 2f, rightElement.size.y / 2f),
            clickable = rightElement.clickable,
            targetAction = rightElement.targetAction
        };

        yield return StartCoroutine(CreateMenuHeaderElement(headerPanel, modifiedRightElement, "RightElement"));
    }

    private IEnumerator CreateMenuHeaderElement(GameObject headerPanel, DefaultHeaderElement elementData, string name)
    {
        GameObject elementObj = new GameObject(name);
        elementObj.transform.SetParent(headerPanel.transform, false);

        RectTransform rt = elementObj.AddComponent<RectTransform>();
        rt.anchorMin = new Vector2(1f, 1f);
        rt.anchorMax = new Vector2(1f, 1f);
        rt.pivot = new Vector2(1f, 1f);
        rt.anchoredPosition = elementData.position;
        rt.sizeDelta = elementData.size;

        Image img = elementObj.AddComponent<Image>();
        img.raycastTarget = elementData.clickable;

        Canvas parentCanvas = headerPanel.GetComponentInParent<Canvas>();
        if (parentCanvas != null && parentCanvas.GetComponent<GraphicRaycaster>() == null)
        {
            parentCanvas.gameObject.AddComponent<GraphicRaycaster>();
        }

        HeaderElementUI headerElement = elementObj.AddComponent<HeaderElementUI>();
        headerElement.Initialize(elementData, HeaderManager.Instance);

        if (!string.IsNullOrEmpty(elementData.imageUrl))
        {
            string fullImageUrl = GeneralConfigManager.Instance.GetDecoratorImageUrl(elementData.imageUrl);

            using (UnityWebRequest www = UnityWebRequestTexture.GetTexture(fullImageUrl))
            {
                yield return www.SendWebRequest();

                if (www.result == UnityWebRequest.Result.Success)
                {
                    Texture2D texture = ((DownloadHandlerTexture)www.downloadHandler).texture;
                    Sprite sprite = Sprite.Create(texture,
                        new Rect(0, 0, texture.width, texture.height),
                        new Vector2(0.5f, 0.5f));
                    headerElement.SetSprite(sprite);
                }
                else
                {
                    Debug.LogError($"[MainMenuManager] Erreur chargement image header: {www.error}");
                }
            }
        }
    }

    private void OnHeaderElementClick(string action)
    {
        Debug.Log($"[MainMenuManager] OnHeaderElementClick: {action}");

        switch (action)
        {
            case "settings":
                if (SettingsManager.Instance == null)
                {
                    GameObject settingsManagerObj = new GameObject("SettingsManager");
                    DontDestroyOnLoad(settingsManagerObj);
                    settingsManagerObj.AddComponent<SettingsManager>();
                    StartCoroutine(WaitForSettingsManagerAndOpen());
                }
                else
                {
                    SettingsManager.Instance.OpenSettings();
                }
                break;

            default:
                Debug.LogWarning($"[MainMenuManager] Action inconnue: {action}");
                break;
        }
    }

    private IEnumerator WaitForSettingsManagerAndOpen()
    {
        yield return null;
        if (SettingsManager.Instance != null)
        {
            SettingsManager.Instance.OpenSettings();
        }
    }

    private IEnumerator CheckLoginStatusAndShowPopup()
    {
        yield return new WaitForSeconds(0.5f);

        // Si l'utilisateur vient de cliquer "Se déconnecter", on FORCE l'affichage de la popup
        // même en iframe (sinon un token parent résiduel peut supprimer la popup).
        if (PlayerPrefs.GetInt("ForceShowLoginPopup", 0) == 1)
        {
            PlayerPrefs.DeleteKey("ForceShowLoginPopup");
            PlayerPrefs.Save();
            Debug.Log("[MainMenuManager] 🔐 ForceShowLoginPopup=1 -> affichage popup");
            ShowLoginPopup();
            yield break;
        }
        
        // Si on est dans une iframe, attendre le token et l'authentification (max 5 secondes)
        if (PostMessageBridge.Instance != null && PostMessageBridge.Instance.IsInIframe)
        {
            Debug.Log("[MainMenuManager] 📡 Dans une iframe - attente du token parent...");
            float timeout = 5f;
            float elapsed = 0f;
            
            while (elapsed < timeout)
            {
                if (UserDataManager.Instance != null && UserDataManager.Instance.IsLoggedIn())
                {
                    Debug.Log($"[MainMenuManager] ✓ Utilisateur connecté après {elapsed:F1}s");
                    break;
                }
                
                if (IframeAuthManager.Instance != null && IframeAuthManager.Instance.IsAuthenticatedViaIframe)
                {
                    Debug.Log($"[MainMenuManager] ✓ Authentifié via iframe après {elapsed:F1}s");
                    break;
                }
                
                bool hasToken = PostMessageBridge.Instance.HasToken || PostMessageBridge.Instance.HasPendingToken;
                bool isAuthenticating = IframeAuthManager.Instance != null && IframeAuthManager.Instance.IsAuthenticating;
                
                if (!hasToken && !isAuthenticating && elapsed > 2f)
                {
                    Debug.Log($"[MainMenuManager] ⚠ Aucun token reçu après {elapsed:F1}s - affichage popup");
                    break;
                }
                
                yield return new WaitForSeconds(0.1f);
                elapsed += 0.1f;
            }
        }

        if (UserDataManager.Instance == null || !UserDataManager.Instance.IsLoggedIn())
        {
            if (IframeAuthManager.Instance != null && IframeAuthManager.Instance.IsAuthenticatedViaIframe)
            {
                Debug.Log("[MainMenuManager] ✓ Utilisateur authentifié via iframe - popup non affichée");
            }
            // Si on est dans une iframe et qu'on a un token (même si la vérification a échoué),
            // ne pas afficher la popup - le token pourrait être valide mais le serveur a un problème temporaire
            else if (PostMessageBridge.Instance != null && PostMessageBridge.Instance.IsInIframe && 
                     (PostMessageBridge.Instance.HasToken || PostMessageBridge.Instance.HasPendingToken))
            {
                Debug.Log("[MainMenuManager] ⚠ Token présent mais vérification échouée - popup non affichée (erreur serveur possible)");
            }
            else
            {
                Debug.Log("[MainMenuManager] ⚠️ Utilisateur non connecté - Affichage du panneau d'identification");
                ShowLoginPopup();
            }
        }
        else
        {
            Debug.Log("[MainMenuManager] ✅ Utilisateur déjà connecté");
        }
    }

    private void ShowLoginPopup()
    {
        LoginPopup existingPopup = FindFirstObjectByType<LoginPopup>();
        if (existingPopup != null) return;

        GameObject popupObj = new GameObject("LoginPopup");
        LoginPopup loginPopup = popupObj.AddComponent<LoginPopup>();
        loginPopup.Initialize(
            onSuccess: () => Debug.Log("[MainMenuManager] ✅ Connexion réussie"),
            onCloseCallback: () => Debug.Log("[MainMenuManager] Popup fermée")
        );
        loginPopup.Show();
    }

    // ========================================
    // PUBLIC METHODS
    // ========================================

    public void RefreshMenu()
    {
        StartCoroutine(LoadGamesFromCSV());
    }

    public void SetColumns(int col)
    {
        columns = Mathf.Max(1, col);
        if (grid != null) grid.constraintCount = columns;
        GenerateGameCards();
        StartCoroutine(UpdateLayoutWithDelay());
    }

    public void SetCellSize(float w, float h)
    {
        cellSize = new Vector2(w, h);
        if (grid != null) grid.cellSize = cellSize;
        GenerateGameCards();
        StartCoroutine(UpdateLayoutWithDelay());
    }

    // ========================================
    // INPUT
    // ========================================

    void Update()
    {
        var k = UnityEngine.InputSystem.Keyboard.current;
        if (k == null) return;

        // ESC pour quitter
        if (k.escapeKey.wasPressedThisFrame)
        {
            Debug.Log("[MainMenuManager] 🚪 ESC - Quitter le jeu");
            QuitApplication();
            return;
        }

        // Debug keys
        if (!debugMode) return;

        if (k.rKey.wasPressedThisFrame) RefreshMenu();
        if (k.digit1Key.wasPressedThisFrame) SetColumns(1);
        if (k.digit2Key.wasPressedThisFrame) SetColumns(2);
        if (k.digit3Key.wasPressedThisFrame) SetColumns(3);
        if (k.digit4Key.wasPressedThisFrame) SetColumns(4);
        if (k.digit5Key.wasPressedThisFrame) SetColumns(5);
        if (k.digit6Key.wasPressedThisFrame) SetColumns(6);
    }

    void QuitApplication()
    {
#if UNITY_EDITOR
        UnityEditor.EditorApplication.isPlaying = false;
#else
        Application.Quit();
#endif
    }
}
