using UnityEngine;
using UnityEngine.UI;
using UnityEngine.InputSystem;
using TMPro;
using System.Collections;
using System.Collections.Generic;

public class SettingsManager : MonoBehaviour
{
    public GameObject settingsPanel;
    public GameObject darkBackground;
    public Button closeButton;
    public Transform tabsContainer;
    public Transform contentContainer;
    public GameObject tabButtonPrefab;
    public List<SettingsTab> tabs = new List<SettingsTab>();
    public Color activeTabColor = new Color(1f, 1f, 1f, 1f);
    public Color inactiveTabColor = new Color(0.7f, 0.7f, 0.7f, 1f);
    
    private List<Button> tabButtons = new List<Button>();
    private static SettingsManager instance;
    private MapManager mapManager; // Référence au MapManager pour le désactiver
    private MainSceneManager mainSceneManager; // Référence au MainSceneManager pour le désactiver
    
    public static SettingsManager Instance => instance;
    
    /// <summary>
    /// Obtient ou crée l'instance du SettingsManager
    /// </summary>
    public static SettingsManager GetOrCreateInstance()
    {
        if (instance == null)
        {
            Debug.Log("[SettingsManager] 🔧 Création automatique de l'instance...");
            GameObject settingsManagerObj = new GameObject("SettingsManager");
            DontDestroyOnLoad(settingsManagerObj);
            instance = settingsManagerObj.AddComponent<SettingsManager>();
        }
        return instance;
    }
    
    void Awake()
    {
        if (instance == null)
        {
            instance = this;
            DontDestroyOnLoad(gameObject); // Rendre le SettingsManager persistant entre les scènes
            Debug.Log("[SettingsManager] ✅ Instance créée et rendue persistante");
            
            // S'abonner aux changements de scène pour nettoyer le panel
            UnityEngine.SceneManagement.SceneManager.sceneLoaded += OnSceneLoaded;
        }
        else
        {
            Destroy(gameObject);
            return;
        }
        
        if (settingsPanel != null)
        {
            settingsPanel.SetActive(false);
        }
        
        if (closeButton != null)
        {
            closeButton.onClick.RemoveAllListeners();
            closeButton.onClick.AddListener(CloseSettings);
        }
        
        if (darkBackground != null)
        {
            Button bgButton = darkBackground.GetComponent<Button>();
            if (bgButton != null)
            {
                bgButton.onClick.RemoveAllListeners();
                bgButton.onClick.AddListener(CloseSettings);
            }
        }
        
        // CORRECTION : Trouver le MapManager pour le désactiver quand le panneau est ouvert
        mapManager = FindFirstObjectByType<MapManager>();
    }
    
    void OnDestroy()
    {
        // Se désabonner des événements de scène
        UnityEngine.SceneManagement.SceneManager.sceneLoaded -= OnSceneLoaded;
    }
    
    /// <summary>
    /// Nettoie le panel des settings quand on change de scène
    /// Ceci évite que des éléments UI du jeu restent attachés au panel
    /// </summary>
    private void OnSceneLoaded(UnityEngine.SceneManagement.Scene scene, UnityEngine.SceneManagement.LoadSceneMode mode)
    {
        Debug.Log($"[SettingsManager] 📦 Nouvelle scène chargée: {scene.name}");
        
        // Détruire le panel existant pour éviter les références corrompues
        // Il sera recréé automatiquement quand on ouvre les settings
        if (settingsPanel != null)
        {
            Debug.Log("[SettingsManager] 🗑️ Destruction du panel lors du changement de scène");
            Destroy(settingsPanel);
            settingsPanel = null;
            darkBackground = null;
            closeButton = null;
            tabs.Clear();
            tabButtons.Clear();
        }
        
        // Réactiver le MapManager si nécessaire (il a peut-être été désactivé)
        mapManager = null;
        mainSceneManager = null;
    }
    
    void Update()
    {
        if (settingsPanel != null && settingsPanel.activeSelf)
        {
            if (Keyboard.current != null && Keyboard.current.escapeKey.wasPressedThisFrame)
            {
                CloseSettings();
            }
        }
    }
    
    public void OpenSettings()
    {
        Canvas mainCanvas = FindMainCanvas();
        if (mainCanvas == null)
        {
            Debug.LogError("[SettingsManager] ❌ Impossible d'ouvrir le panneau : aucun Canvas trouvé !");
            return;
        }
        
        // TOUJOURS recréer le panneau pour éviter les problèmes de layout entre scènes
        // Détruire l'ancien panneau s'il existe
        if (settingsPanel != null)
        {
            Debug.Log("[SettingsManager] ⚠️ Destruction de l'ancien panneau pour recréation complète");
            Destroy(settingsPanel);
            settingsPanel = null;
            darkBackground = null;
            tabs.Clear();
            tabButtons.Clear();
        }
        
        // Créer le panneau sur le bon Canvas (coroutine)
        StartCoroutine(OpenSettingsCoroutine());
    }
    
    private System.Collections.IEnumerator OpenSettingsCoroutine()
    {
        // Attendre que le panneau soit créé
        yield return StartCoroutine(CreateSettingsPanelIfNeededCoroutine());
        
        if (settingsPanel == null)
        {
            Debug.LogError("[SettingsManager] Impossible de créer le panneau des paramètres !");
            yield break;
        }
        
        Canvas mainCanvas = FindMainCanvas();
        
        // S'assurer que le panneau est bien sur le bon Canvas
        if (mainCanvas != null && settingsPanel.transform.parent != mainCanvas.transform)
        {
            settingsPanel.transform.SetParent(mainCanvas.transform, false);
            Debug.Log($"[SettingsManager] ✅ Panneau repositionné sur le Canvas: {mainCanvas.name}");
        }

        settingsPanel.SetActive(true);
        
        // Forcer une mise à jour du layout après activation
        Canvas.ForceUpdateCanvases();
        if (settingsPanel != null)
        {
            RectTransform panelRect = settingsPanel.GetComponent<RectTransform>();
            if (panelRect != null)
            {
                LayoutRebuilder.ForceRebuildLayoutImmediate(panelRect);
            }
        }
        
        Debug.Log($"[SettingsManager] Panneau activé, nombre d'onglets: {tabs.Count}");
        
        CreateTabButtons();
        
        // Attendre une frame pour que tout soit initialisé
        yield return null;
        
        // S'assurer que les couleurs des boutons sont mises à jour avant d'activer l'onglet
        if (tabsContainer != null)
        {
            Debug.Log("[SettingsManager] Mise à jour des couleurs des boutons avant activation de l'onglet SCORES");
            UpdateTabButtonColors(0); // Mettre à jour les couleurs pour l'onglet 0 (SCORES)
        }
        else
        {
            Debug.LogWarning("[SettingsManager] ⚠️ tabsContainer est null, impossible de mettre à jour les couleurs");
        }
        
        Debug.Log("[SettingsManager] Activation de l'onglet SCORES (ShowTab(0))");
        ShowTab(0);
        
        // Vérifier que le ScoresTab est bien activé
        if (tabs.Count > 0 && tabs[0] != null)
        {
            Debug.Log($"[SettingsManager] ScoresTab après ShowTab(0): active={tabs[0].gameObject.activeSelf}, activeInHierarchy={tabs[0].gameObject.activeInHierarchy}");
        }
        
        // Désactiver le MapManager s'il existe (pour la scène Map)
        mapManager = FindFirstObjectByType<MapManager>();
        if (mapManager != null)
        {
            mapManager.enabled = false;
            
            // **FIX : Désactiver le ScrollRect de la map pour bloquer les interactions**
            if (mapManager.scrollViewObject != null)
            {
                ScrollRect mapScrollRect = mapManager.scrollViewObject.GetComponent<ScrollRect>();
                if (mapScrollRect != null)
                {
                    mapScrollRect.enabled = false;
                    Debug.Log("[SettingsManager] MapManager ScrollRect désactivé - interactions avec map bloquées");
                }
            }
            
            Debug.Log("[SettingsManager] MapManager désactivé");
        }
        
        // Désactiver le MainSceneManager s'il existe (pour la scène Main)
        mainSceneManager = FindFirstObjectByType<MainSceneManager>();
        if (mainSceneManager != null)
        {
            mainSceneManager.enabled = false;
            Debug.Log("[SettingsManager] MainSceneManager désactivé");
        }
    }

    /// <summary>
    /// Trouve le MainCanvas principal de la scène
    /// </summary>
    private Canvas FindMainCanvas()
    {
        // Trouver le MainCanvas spécifiquement, pas n'importe quel Canvas
        Canvas mainCanvas = null;
        Canvas[] allCanvases = FindObjectsByType<Canvas>(FindObjectsSortMode.None);
        
        // Priorité 1 : Chercher "MainCanvas" (scène Main)
        foreach (Canvas canvas in allCanvases)
        {
            if (canvas.name == "MainCanvas" && canvas.renderMode == RenderMode.ScreenSpaceOverlay && canvas.transform.parent == null)
            {
                mainCanvas = canvas;
                Debug.Log($"[SettingsManager] ✅ MainCanvas trouvé: {canvas.name}");
                break;
            }
        }
        
        // Priorité 2 : Chercher "Canvas" (scène Map) - pas PopupContainer
        if (mainCanvas == null)
        {
            foreach (Canvas canvas in allCanvases)
            {
                if (canvas.name == "Canvas" && canvas.renderMode == RenderMode.ScreenSpaceOverlay && 
                    canvas.transform.parent == null && canvas.gameObject.name != "PopupContainer")
                {
                    mainCanvas = canvas;
                    Debug.Log($"[SettingsManager] ✅ Canvas de scène Map trouvé: {canvas.name}");
                    break;
                }
            }
        }
        
        // Priorité 3 : Prendre le premier Canvas racine en ScreenSpaceOverlay (pas PopupContainer)
        if (mainCanvas == null)
        {
            foreach (Canvas canvas in allCanvases)
            {
                if (canvas.transform.parent == null && canvas.renderMode == RenderMode.ScreenSpaceOverlay && 
                    canvas.gameObject.name != "PopupContainer" && !canvas.gameObject.name.Contains("SettingsPanel"))
                {
                    mainCanvas = canvas;
                    Debug.Log($"[SettingsManager] ✅ Canvas racine trouvé: {canvas.name}");
                    break;
                }
            }
        }
        
        if (mainCanvas == null)
        {
            Debug.LogError("[SettingsManager] ❌ Aucun Canvas principal trouvé !");
        }
        
        return mainCanvas;
    }

    /// <summary>
    /// Crée le panneau des paramètres si nécessaire (coroutine pour attendre le chargement des images)
    /// </summary>
    private System.Collections.IEnumerator CreateSettingsPanelIfNeededCoroutine()
    {
        Debug.Log("[SettingsManager] 🔧 Création du panneau des paramètres...");
        
        // FORCER la destruction du panneau existant s'il y en a un
        if (settingsPanel != null)
        {
            Debug.Log("[SettingsManager] ⚠️ Destruction du panneau existant pour le recréer avec nouveau fond sombre");
            Destroy(settingsPanel);
            settingsPanel = null;
            darkBackground = null;
        }
        
        Canvas currentCanvas = FindMainCanvas();
        
        if (currentCanvas == null)
        {
            Debug.LogError("[SettingsManager] ❌ Aucun Canvas principal trouvé pour créer le panneau des paramètres");
            yield break;
        }

        Debug.Log($"[SettingsManager] ✅ Canvas principal trouvé: {currentCanvas.name}");

        // Chercher SettingsPanelBuilder dans la scène
        SettingsPanelBuilder builder = FindFirstObjectByType<SettingsPanelBuilder>();
        if (builder != null)
        {
            Debug.Log("[SettingsManager] SettingsPanelBuilder trouvé dans la scène");
            Debug.Log("[SettingsManager] Création du panneau via SettingsPanelBuilder existant...");
            builder.BuildSettingsPanel(); // Lance une coroutine
            
            // Attendre que le panneau soit créé (la coroutine BuildSettingsPanelCoroutine doit se terminer)
            yield return new WaitForSeconds(0.1f); // Attendre un peu pour le chargement des images
            
            // Attendre jusqu'à ce que le panneau soit créé (max 5 secondes)
            float timeout = 5f;
            float elapsed = 0f;
            while (settingsPanel == null && elapsed < timeout)
            {
                yield return new WaitForSeconds(0.1f);
                elapsed += 0.1f;
            }
            
            if (settingsPanel == null)
            {
                Debug.LogError("[SettingsManager] ❌ Le panneau n'a pas été créé après l'attente");
                yield break;
            }
            
            Debug.Log("[SettingsManager] ✅ Panneau créé avec succès");
            yield break;
        }

        // Si pas de builder, créer le panneau manuellement
        Debug.Log("[SettingsManager] ⚙️ Création manuelle du panneau des paramètres");
        
        // Créer un GameObject temporaire avec SettingsPanelBuilder pour construire le panneau
        GameObject builderObj = new GameObject("SettingsPanelBuilder");
        builderObj.transform.SetParent(currentCanvas.transform, false);
        SettingsPanelBuilder tempBuilder = builderObj.AddComponent<SettingsPanelBuilder>();
        
        // S'assurer que le parentCanvas est défini dans le builder
        tempBuilder.SetParentCanvas(currentCanvas);
        
        // Forcer la création (lance une coroutine)
        Debug.Log("[SettingsManager] Appel de BuildSettingsPanel()...");
        tempBuilder.BuildSettingsPanel();
        
        // Attendre que le panneau soit créé (max 5 secondes)
        float timeout2 = 5f;
        float elapsed2 = 0f;
        while (settingsPanel == null && elapsed2 < timeout2)
        {
            yield return new WaitForSeconds(0.1f);
            elapsed2 += 0.1f;
        }
        
        if (settingsPanel == null)
        {
            Debug.LogError("[SettingsManager] ❌ Le panneau n'a pas été créé après l'attente");
            yield break;
        }
        
        Debug.Log("[SettingsManager] ✅ Panneau créé avec succès");
    }

    private System.Collections.IEnumerator WaitForPanelAfterCreation()
    {
        yield return null; // Attendre une frame
        
        if (settingsPanel == null)
        {
            Debug.LogError("[SettingsManager] ❌ Le panneau n'a pas été créé après l'attente");
        }
        else
        {
            Debug.Log("[SettingsManager] ✅ Panneau créé avec succès, tentative d'ouverture...");
            // Réessayer d'ouvrir le panneau maintenant qu'il est créé
            OpenSettings();
        }
    }
    
    public void CloseSettings()
    {
        Debug.Log("[SettingsManager] CloseSettings() appelé");
        
        if (settingsPanel != null && settingsPanel.activeSelf)
        {
            Debug.Log("[SettingsManager] Fermeture du panneau en cours...");
            StartCoroutine(CloseWithFade());
        }
        else
        {
            Debug.LogWarning("[SettingsManager] ⚠️ Tentative de fermeture mais le panneau est déjà fermé ou null");
        }
    }
    
    IEnumerator CloseWithFade()
    {
        Debug.Log("[SettingsManager] CloseWithFade() démarré");
        
        if (settingsPanel == null)
        {
            Debug.LogError("[SettingsManager] ❌ settingsPanel est null dans CloseWithFade()");
            yield break;
        }
        
        CanvasGroup cg = settingsPanel.GetComponent<CanvasGroup>();
        if (cg == null)
        {
            cg = settingsPanel.AddComponent<CanvasGroup>();
        }
        
        float elapsed = 0f;
        float duration = 0.2f;
        
        while (elapsed < duration)
        {
            // Vérifier que le CanvasGroup existe toujours avant de modifier alpha
            if (cg == null || settingsPanel == null)
            {
                yield break;
            }
            
            elapsed += Time.deltaTime;
            cg.alpha = Mathf.Lerp(1f, 0f, elapsed / duration);
            yield return null;
        }
        
        // Vérifier à nouveau avant de finaliser
        if (cg != null && settingsPanel != null)
        {
            cg.alpha = 1f;
            
            // **FIX OOM : Nettoyer les ressources des onglets avant de fermer**
            foreach (SettingsTab tab in tabs)
            {
                if (tab != null && tab is QuetesTab)
                {
                    QuetesTab quetesTab = tab as QuetesTab;
                    quetesTab.CleanupOnPanelClose();
                }
            }
            
            settingsPanel.SetActive(false);
            Debug.Log("[SettingsManager] ✅ Panneau désactivé");
        }
        
        // Réactiver le MapManager s'il existe
        if (mapManager != null)
        {
            mapManager.enabled = true;
            
            // **FIX : Réactiver le ScrollRect de la map pour restaurer les interactions**
            if (mapManager.scrollViewObject != null)
            {
                ScrollRect mapScrollRect = mapManager.scrollViewObject.GetComponent<ScrollRect>();
                if (mapScrollRect != null)
                {
                    mapScrollRect.enabled = true;
                    Debug.Log("[SettingsManager] MapManager ScrollRect réactivé - interactions avec map restaurées");
                }
            }
            
            Debug.Log("[SettingsManager] MapManager réactivé");
        }
        
        // Réactiver le MainSceneManager s'il existe
        if (mainSceneManager != null)
        {
            mainSceneManager.enabled = true;
            Debug.Log("[SettingsManager] MainSceneManager réactivé");
        }
        else
        {
            // Chercher à nouveau au cas où
            mainSceneManager = FindFirstObjectByType<MainSceneManager>();
            if (mainSceneManager != null)
            {
                mainSceneManager.enabled = true;
                Debug.Log("[SettingsManager] MainSceneManager réactivé (trouvé à nouveau)");
            }
        }
    }
    
    void CreateTabButtons()
    {
        // Les boutons sont déjà créés par SettingsPanelBuilder
        // Cette méthode n'est plus nécessaire mais on la garde pour compatibilité
        if (tabButtons.Count > 0) return;
        
        // Si tabsContainer est null, sortir
        if (tabsContainer == null)
        {
            Debug.LogWarning("[SettingsManager] tabsContainer est null, impossible de créer les boutons");
            return;
        }
        
        // Récupérer les boutons existants créés par SettingsPanelBuilder
        foreach (Transform child in tabsContainer)
        {
            Button button = child.GetComponent<Button>();
            if (button != null)
            {
                tabButtons.Add(button);
            }
        }
        
        Debug.Log($"[SettingsManager] {tabButtons.Count} boutons d'onglets trouvés");
    }
    

    
    public void RegisterTab(SettingsTab tab)
    {
        if (!tabs.Contains(tab))
        {
            tabs.Add(tab);
        }
    }


    public void ShowTab(int index)
    {
        if (index < 0 || index >= tabs.Count)
        {
            Debug.LogWarning($"[SettingsManager] ShowTab({index}) - Index invalide (tabs.Count: {tabs.Count})");
            return;
        }
        
        for (int i = 0; i < tabs.Count; i++)
        {
            if (tabs[i] != null && tabs[i].gameObject != null)
            {
                bool shouldBeActive = (i == index);
                tabs[i].gameObject.SetActive(shouldBeActive);
            }
            else
            {
                Debug.LogError($"[SettingsManager] ❌ Onglet {i} est null !");
            }
        }
        
        // Mettre à jour les couleurs des boutons d'onglets
        UpdateTabButtonColors(index);
    }

    void UpdateTabButtonColors(int activeIndex)
    {
        if (tabsContainer == null)
        {
            Debug.LogWarning("[SettingsManager] ⚠️ tabsContainer est null dans UpdateTabButtonColors, impossible de mettre à jour les couleurs");
            return;
        }
        
        // Récupérer les couleurs depuis la config
        var config = GeneralConfigManager.Instance?.GetConfig();
        var tabsConfig = config?.settingsPanelConfig?.tabs;
        
        Color normalTextColor = new Color(0.47f, 0.4f, 0.37f, 1f);
        Color selectedBgColor = new Color(0.64f, 0.25f, 1f, 1f);
        Color selectedTextColor = Color.white;
        
        if (tabsConfig?.normal != null && !string.IsNullOrEmpty(tabsConfig.normal.textColor))
        {
            ColorUtility.TryParseHtmlString(tabsConfig.normal.textColor, out normalTextColor);
        }
        if (tabsConfig?.selected != null)
        {
            if (!string.IsNullOrEmpty(tabsConfig.selected.backgroundColor))
            {
                if (ColorUtility.TryParseHtmlString(tabsConfig.selected.backgroundColor, out Color parsedColor))
                {
                    selectedBgColor = parsedColor;
                    Debug.Log($"[SettingsManager] Couleur onglet sélectionné: {tabsConfig.selected.backgroundColor} -> {selectedBgColor}");
                }
            }
            if (!string.IsNullOrEmpty(tabsConfig.selected.textColor))
            {
                ColorUtility.TryParseHtmlString(tabsConfig.selected.textColor, out selectedTextColor);
            }
        }
        
        Debug.Log($"[SettingsManager] UpdateTabButtonColors({activeIndex}) - tabsContainer: {tabsContainer != null}, childCount: {(tabsContainer != null ? tabsContainer.childCount : 0)}");
        
        int buttonIndex = 0;
        foreach (Transform child in tabsContainer)
        {
            Button button = child.GetComponent<Button>();
            Image image = child.GetComponent<Image>();
            TextMeshProUGUI text = child.GetComponentInChildren<TextMeshProUGUI>();
            
            if (button != null && image != null)
            {
                if (buttonIndex == activeIndex)
                {
                    // Onglet sélectionné : fond coloré, texte blanc
                    image.color = selectedBgColor;
                    if (text != null)
                    {
                        text.color = selectedTextColor;
                        Debug.Log($"[SettingsManager] ✅ Onglet {buttonIndex} ({child.name}) : sélectionné");
                    }
                }
                else
                {
                    // Onglets non sélectionnés : fond transparent
                    image.color = Color.clear;
                    if (text != null)
                    {
                        text.color = normalTextColor;
                    }
                }
                buttonIndex++;
            }
            else
            {
                Debug.LogWarning($"[SettingsManager] ⚠️ Bouton ou Image introuvable pour l'enfant {child.name}");
            }
        }
        
        if (buttonIndex == 0)
        {
            Debug.LogWarning("[SettingsManager] ⚠️ Aucun bouton d'onglet trouvé dans tabsContainer");
        }
    }
    
    public int GetActiveTabIndex()
    {
        if (tabs == null || tabs.Count == 0) return -1;
        
        for (int i = 0; i < tabs.Count; i++)
        {
            if (tabs[i] != null && tabs[i].gameObject.activeSelf)
            {
                return i;
            }
        }
        return 0; // Par défaut, retourner le premier onglet
    }
}