using System;
using System.Collections.Generic;
using UnityEngine;

/// <summary>
/// Tracke et détruit explicitement les ressources runtime (Texture2D, Sprite, RenderTexture, etc.).
/// Objectif: éviter les fuites mémoire et les abort(OOM) en WebGL.
/// </summary>
public sealed class RuntimeResourceTracker
{
    private readonly List<UnityEngine.Object> _objects = new List<UnityEngine.Object>(256);

    /// <summary>Ajoute un objet Unity à la liste de tracking (si non-null).</summary>
    public T Track<T>(T obj) where T : UnityEngine.Object
    {
        if (obj != null) _objects.Add(obj);
        return obj;
    }

    /// <summary>
    /// Track un couple (texture + sprite) qui ont été créés ensemble.
    /// </summary>
    public void Track(Texture2D tex, Sprite sp)
    {
        if (sp != null) _objects.Add(sp);
        if (tex != null) _objects.Add(tex);
    }

    /// <summary>Track un RenderTexture.</summary>
    public RenderTexture Track(RenderTexture rt)
    {
        if (rt != null) _objects.Add(rt);
        return rt;
    }

    /// <summary>
    /// Détruit tout ce qui a été tracké.
    /// IMPORTANT: ne pas appeler Resources.UnloadUnusedAssets / GC.Collect ici (coûteux et instable en WebGL).
    /// </summary>
    public void Cleanup(bool destroyImmediateInEditor = false)
    {
        if (_objects.Count == 0) return;

        for (int i = 0; i < _objects.Count; i++)
        {
            var o = _objects[i];
            if (o == null) continue;

            // RenderTexture: Release avant destroy
            if (o is RenderTexture rt)
            {
                try { rt.Release(); } catch { }
            }

#if UNITY_EDITOR
            if (destroyImmediateInEditor)
                UnityEngine.Object.DestroyImmediate(o);
            else
                UnityEngine.Object.Destroy(o);
#else
            UnityEngine.Object.Destroy(o);
#endif
        }

        _objects.Clear();
    }

    /// <summary>Nombre d'objets trackés (debug).</summary>
    public int Count => _objects.Count;
}


