#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Script pour convertir les URLs complètes en noms de fichiers dans les JSON
"""

import json
import os
import glob

# Définir les chemins de base à remplacer
BASE_PATHS = {
    'https://unjoursansassurance.studioplc.tech/demo_assets/images/': '',
    'https://unjoursansassurance.studioplc.tech/demo_assets/videos/': '',
    'https://unjoursansassurance.studioplc.tech/demo_assets/UI/': '',
    'https://unjoursansassurance.studioplc.tech/demo_assets/dialogues/': '',
    'https://unjoursansassurance.studioplc.tech/demo_assets/questions/': '',
    'https://unjoursansassurance.studioplc.tech/demo_assets/json/maps/': '',
    'https://unjoursansassurance.studioplc.tech/videos/': ''
}

def replace_urls_in_string(text):
    """Remplace les URLs complètes par les noms de fichiers"""
    if not isinstance(text, str):
        return text
    
    for base_url, replacement in BASE_PATHS.items():
        if text.startswith(base_url):
            return text.replace(base_url, replacement)
    return text

def process_json_value(value):
    """Traite récursivement les valeurs JSON"""
    if isinstance(value, str):
        return replace_urls_in_string(value)
    elif isinstance(value, dict):
        return {k: process_json_value(v) for k, v in value.items()}
    elif isinstance(value, list):
        return [process_json_value(item) for item in value]
    else:
        return value

def convert_json_file(filepath):
    """Convertit un fichier JSON"""
    print(f"Traitement de {os.path.basename(filepath)}...")
    
    try:
        with open(filepath, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        # Traiter toutes les valeurs récursivement
        converted_data = process_json_value(data)
        
        # Sauvegarder le fichier modifié
        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(converted_data, f, indent=4, ensure_ascii=False)
        
        print(f"  [OK] {os.path.basename(filepath)} converti")
        return True
    except Exception as e:
        print(f"  [ERROR] Erreur: {e}")
        return False

def main():
    """Fonction principale"""
    json_dir = os.path.join(os.path.dirname(__file__), 'json')
    
    print("="*60)
    print("Conversion des URLs dans les fichiers JSON")
    print("="*60)
    print()
    
    # Traiter tous les fichiers JSON sauf convert_json_paths.py
    all_json_files = []
    
    # Ajouter tous les fichiers map-*.json
    all_json_files.extend(glob.glob(os.path.join(json_dir, 'map-*.json')))
    
    # Ajouter tous les fichiers Q*.json
    all_json_files.extend(glob.glob(os.path.join(json_dir, 'Q*.json')))
    
    # Ajouter les fichiers de configuration
    all_json_files.extend(glob.glob(os.path.join(json_dir, '*-config.json')))
    all_json_files.extend(glob.glob(os.path.join(json_dir, '*-registry.json')))
    
    # Dédupliquer
    all_json_files = list(set(all_json_files))
    
    success_count = 0
    for filepath in sorted(all_json_files):
        if convert_json_file(filepath):
            success_count += 1
    
    print()
    print("="*60)
    print(f"Terminé : {success_count}/{len(all_json_files)} fichiers convertis")
    print("="*60)

if __name__ == '__main__':
    main()

